package coremodules

import (
	"fmt"
	"os"

	"github.com/heysion/deepin-system-update-tools/internal/config"
	"github.com/heysion/deepin-system-update-tools/internal/config/cache"
	"github.com/heysion/deepin-system-update-tools/pkg/log"
	"github.com/heysion/deepin-system-update-tools/pkg/utils/ecode"
	"github.com/spf13/cobra"
)

var (
	ConfigCfg    string
	DebugVerbose bool
	FixpkgConfig config.CoreConfig
	CacheCfg     cache.CacheConfig
	CheckRetMsg  ecode.RetMsg
)

// rootCmd represents the base command when called without any subcommands
var rootCmd = &cobra.Command{
	Use:   "deepin system fix package depends tools",
	Short: "deepin system fix package depends tools",
	Long: `deepin system fix package depends tools:

deepin system fix package is a linux system package depends fix tool that depends on the dpkg
package manager.`,
	// Uncomment the following line if your bare application
	// has an action associated with it:
	//      Run: func(cmd *cobra.Command, args []string) { },
	PersistentPreRun: func(cmd *cobra.Command, args []string) {
		if DebugVerbose {
			log.SetDebugEnabled()
		}
		log.Debugf("load config with disk")
		if err := FixpkgConfig.LoaderCfg(ConfigCfg); err != nil {
			log.Errorf("%+v", err)
			CheckRetMsg.ExitOutput(ecode.CHK_INVALID_INPUT, ecode.CHK_METAINFO_FILE_ERROR, fmt.Sprintf("%+v", err))
		}

		if err := FixpkgConfig.LoaderCache(&CacheCfg); err != nil {
			log.Errorf("%+v", err)
			CheckRetMsg.ExitOutput(ecode.CHK_INVALID_INPUT, ecode.CHK_METAINFO_FILE_ERROR, fmt.Sprintf("%+v", err))
		}
		// log.Debugf("%+v", CacheCfg)
	},
	PersistentPostRun: func(cmd *cobra.Command, args []string) {
		// flush config to disk
		log.Debugf("flush config to disk")

		if err := FixpkgConfig.UpdateCache(&CacheCfg); err != nil {
			log.Errorf("%+v", err)
			CheckRetMsg.PushExtMsg(fmt.Sprintf("flush cache:%+v", err))

		} else if err := FixpkgConfig.UpdateCfg(ConfigCfg); err != nil {
			log.Errorf("%+v", err)
			CheckRetMsg.PushExtMsg(fmt.Sprintf("flush config:%+v", err))

		}
		if CheckRetMsg.Code != 0 {
			CheckRetMsg.RetMsgToJson()
			os.Exit(int(CheckRetMsg.Code))
		}

	},
}

// Execute adds all child commands to the root command and sets flags appropriately.
// This is called by main.main(). It only needs to happen once to the rootCmd.
func Execute() {
	if err := rootCmd.Execute(); err != nil {
		fmt.Println(err)
		os.Exit(1)
	}
}

func init() {
	rootCmd.PersistentFlags().StringVarP(&ConfigCfg, "config", "c", "/var/lib/deepin-system-update/config.yaml", "config file")
	rootCmd.PersistentFlags().BoolVarP(&DebugVerbose, "debug", "d", false, "debug mode")

	cobra.OnInitialize()

}
