% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/buttons.R
\name{input_task_button}
\alias{input_task_button}
\alias{update_task_button}
\title{Button for launching longer-running operations}
\usage{
input_task_button(
  id,
  label,
  ...,
  icon = NULL,
  label_busy = "Processing...",
  icon_busy = rlang::missing_arg(),
  type = "primary",
  auto_reset = TRUE
)

update_task_button(id, ..., state = NULL, session = get_current_session())
}
\arguments{
\item{id}{The \code{input} slot that will be used to access the value.}

\item{label}{The label of the button while it is in ready (clickable) state;
usually a string.}

\item{...}{Named arguments become attributes to include on the \verb{<button>}
element.}

\item{icon}{An optional icon to display next to the label while the button is
in ready state. See \code{\link[fontawesome:fa_i]{fontawesome::fa_i()}}.}

\item{label_busy}{The label of the button while it is busy.}

\item{icon_busy}{The icon to display while the button is busy. By default,
\code{fontawesome::fa_i("refresh", class = "fa-spin", "aria-hidden" = "true")}
is used, which displays a spinning "chasing arrows" icon. You can create
spinning icons out of other Font Awesome icons by using the same
expression, but replacing \code{"refresh"} with a different icon name. See
\code{\link[fontawesome:fa_i]{fontawesome::fa_i()}}.}

\item{type}{One of the Bootstrap theme colors (\code{"primary"}, \code{"default"},
\code{"secondary"}, \code{"success"}, \code{"danger"}, \code{"warning"}, \code{"info"}, \code{"light"},
\code{"dark"}), or \code{NULL} to leave off the Bootstrap-specific button CSS classes
altogether.}

\item{auto_reset}{If \code{TRUE} (the default), automatically put the button back
in "ready" state after its click is handled by the server.}

\item{state}{If \code{"busy"}, put the button into busy/disabled state. If
\code{"ready"}, put the button into ready/enabled state.}

\item{session}{The \code{session} object; using the default is recommended.}
}
\description{
\code{input_task_button} is a button that can be used in conjuction with
\code{\link[shiny:bindEvent]{shiny::bindEvent()}} (or the older \code{\link[shiny:observeEvent]{shiny::eventReactive()}} and
\code{\link[shiny:observeEvent]{shiny::observeEvent()}} functions) to trigger actions or recomputation.

It is similar to \code{\link[shiny:actionButton]{shiny::actionButton()}}, except it prevents the user from
clicking when its operation is already in progress.

Upon click, it automatically displays a customizable progress message and
disables itself; and after the server has dealt with whatever reactivity is
triggered from the click, the button automatically reverts to its original
appearance and re-enables itself.
}
\section{Manual button reset}{

In some advanced use cases, it may be necessary to keep a task button in its
busy state even after the normal reactive processing has completed. Calling
\code{update_task_button(id, state = "busy")} from the server will opt out of any
currently pending reset for a specific task button. After doing so, the
button can be re-enabled by calling \code{update_task_button(id, state = "ready")}
after each click's work is complete.

You can also pass an explicit \code{auto_reset = FALSE} to \code{input_task_button()},
which means that button will \emph{never} be automatically re-enabled and will
require \code{update_task_button(id, state = "ready")} to be called each time.

Note that, as a general rule, Shiny's \code{update} family of functions do not
take effect at the instant that they are called, but are held until the end
of the current reactive cycle. So if you have many different reactive
calculations and outputs, you don't have to be too careful about when you
call \code{update_task_button(id, state = "ready")}, as the button on the client
will not actually re-enable until the same moment that all of the updated
outputs simultaneously sent to the client.
}

\section{Server value}{

An integer of class \code{"shinyActionButtonValue"}. This class differs from
ordinary integers in that a value of 0 is considered "falsy".
This implies two things:
\itemize{
\item Event handlers (e.g., \code{\link[shiny:observeEvent]{shiny::observeEvent()}}, \code{\link[shiny:observeEvent]{shiny::eventReactive()}}) won't execute on initial load.
\item Input validation (e.g., \code{\link[shiny:req]{shiny::req()}}, \code{\link[shiny:validate]{shiny::need()}}) will fail on initial load.
}
}

\examples{
\dontshow{if (interactive()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(shiny)
library(bslib)

ui <- page_sidebar(
  sidebar = sidebar(
    open = "always",
    input_task_button("resample", "Resample"),
  ),
  verbatimTextOutput("summary")
)

server <- function(input, output, session) {
  sample <- eventReactive(input$resample, ignoreNULL=FALSE, {
    Sys.sleep(2)  # Make this artificially slow
    rnorm(100)
  })

  output$summary <- renderPrint({
    summary(sample())
  })
}

shinyApp(ui, server)
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=bind_task_button]{bind_task_button()}}
}
