/***********************************************************************************
    begin                : Die Sep 16 2003
    copyright            : (C) 2003 by Jeroen Wijnhout (wijnhout@science.uva.nl)
			       2009 by Stefan Hepp (stefan@stefant.org)

    This file is taken from the Kile project and modified to work without Qt on
    the cmdline by Stefan Hepp (stefan@stefant.org).
 ***********************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef LATEXOUTPUTFILTER_H
#define LATEXOUTPUTFILTER_H

#include "outputfilter.h"
#include "outputinfo.h"

#include <cstdio>
#include <string>
#include <stack>

/**
 * An object of this class is used to parse the output messages
 * generated by a TeX/LaTeX-compiler.

 * @author Thorsten Lck
 * @author Jeroen Wijnhout
 * @author Stefan Hepp
 */

class LOFStackItem
{
public:
	explicit LOFStackItem(const std::string& file = "", bool sure = false) : m_file(file), m_reliable(sure) {}

	const std::string & file() const { return m_file; }
	void setFile(const std::string & file) { m_file = file; }

	bool reliable() const { return m_reliable; }
	void setReliable(bool sure) { m_reliable = sure; }

private:
	std::string m_file;
	bool m_reliable;
};

class LatexOutputFilter : public OutputFilter
{
    public:
        LatexOutputFilter(const std::string& source, const std::string& logfile, int verbose, bool nobadboxes, bool quiet);
        ~LatexOutputFilter();

        virtual bool run(FILE *out);

	enum {Start = 0, FileName, FileNameHeuristic, Error, Warning, BadBox, LineNumber};

    protected:
        /**
        Parses the given line for the start of new files or the end of
        old files.
        */
        void updateFileStack(const std::string &strLine, short & dwCookie);
	void updateFileStackHeuristic(const std::string &strLine, short & dwCookie);

        /**
        Forwards the currently parsed item to the item list.
        */
        void flushCurrentItem();

        // overridings
    public:
        /** Return number of errors etc. found in log-file. */
	void getErrorCount(int *errors, int *warnings, int *badboxes);
	void clearErrorCount() { m_nErrors=m_nWarnings=m_nBadBoxes=0 ; }

    protected:
	virtual short parseLine(const std::string & strLine, short dwCookie);

	bool detectError(const std::string & strLine, short &dwCookie);
	bool detectWarning(const std::string & strLine, short &dwCookie);
	bool detectBadBox(const std::string & strLine, short &dwCookie);
	bool detectLaTeXLineNumber(std::string & warning, short & dwCookie, size_t len);
	bool detectBadBoxLineNumber(std::string & strLine, short & dwCookie, size_t len);

	bool fileExists(const std::string & name);

	/** 
	Check if we need to add a space when we append a message to the last line 
	 */
	bool needsSpace();

        // types
    protected:
        /**
        These constants are describing, which item types is currently
        parsed.
        */
        enum tagCookies
        {
            itmNone = 0,
            itmError,
            itmWarning,
            itmBadBox
        };

        // attributes
    private:
        /** number or errors detected */
        int m_nErrors;

        /** number of warning detected */
        int m_nWarnings;

        /** number of bad boxes detected */
        int m_nBadBoxes;

	int m_nParens;

	/** Length of the previous line, to check if we need a space in the next line */
	size_t m_nLastLineLength;

	bool m_nobadboxes;
	bool m_quiet;

        /**
        Stack containing the files parsed by the compiler. The top-most
        element is the currently parsed file.
        */
        std::stack<LOFStackItem> m_stackFile;

        /** The item currently parsed. */
        LatexOutputInfo m_currentItem;

};
#endif
